import { Injectable } from '@angular/core';
import { NgRedux } from 'ng2-redux';
import { IAppState } from '../store';
import {
  BUILD_TAB_DATA,
  UPDATE_CONCEPT_CARD_TAB_SELECTION
} from '../actions';

import { TEMPLATE_MAPPINGS } from './fixtures/template-override';
import { 
  startingTabIndex
} from '../system-config';

@Injectable()
export class TemplatingServices {
  // Types of components we will have
  // List: listed component of single value for each with URI embedded if one exists
  // DoubleColumnList: This will have associated label for each column and 
  // values in column 1 and column 2 - if there are uri for each value, it
  // automatically be embedded
  mappings = TEMPLATE_MAPPINGS;

  constructor(
    private ngRedux: NgRedux<IAppState>) {
  }

  getState() {
    return this.ngRedux.getState();
  }

  setTemplateStates() {
    this.buildTabDataForCurrentConcept();
  }

  setActiveTab(tabName) {
    this.ngRedux.dispatch({
      type: UPDATE_CONCEPT_CARD_TAB_SELECTION,
      payload: tabName
    })
  }

  buildTabDataForCurrentConcept() {
    // Need to build the following object into state redux;
    // templateTabData: {
    //   "nameOfTab": {
    //     tabIndex: 6
    //   }
    // }
    const conceptCardData = this.getState().conceptCard.conceptCardData;
    let mapping = [];
    if (this.currentConceptHasOverride()) {
      mapping = this.mappings[conceptCardData.ontologyName];
    } else {
      mapping = this.mappings['DEFAULT'];
    }
    let tabData = {};
    for (let i = 0; i < mapping.length; i++) {
      if (i===0) {
        this.setActiveTab(mapping[i]['tablink']);
      }
      tabData[mapping[i]['tablink']] = {
        tabIndex: startingTabIndex + i
      }
    }
    this.ngRedux.dispatch({
      type: BUILD_TAB_DATA,
      payload: tabData
    });
  }

  getTabIndex(tabName) {
    const templateTabData = this.getState().conceptCard.templateTabData;
    return templateTabData[tabName].tabIndex;
  }

  getTabIndexForActiveTab() {
    const activeTab = this.getState().conceptCard.conceptCardState.selectedTab;
    const templateTabData = this.getState().conceptCard.templateTabData; 
    return templateTabData[activeTab].tabIndex;
  }

  getColumnSize(size) {
    return 'col-' + size;
  }

  currentConceptHasOverride() {
    const conceptCardData = this.getState().conceptCard.conceptCardData;
    const mapping = this.mappings[conceptCardData.ontologyName];
    return mapping !== undefined && mapping !== null;
  }

  // We will call this based on the concept card ontologySource provided
  // Provided through API
  getMappingFor(ontologySource, mappings = {}) {
    return this.mappings[ontologySource];
  }

  getMappingForCurrentConcept() {
    const state = this.getState();
    const conceptCardData = state.conceptCard.conceptCardData;
    if (this.currentConceptHasOverride()) {
      return this.mappings[conceptCardData.ontologyName];
    } else {
      return this.mappings['DEFAULT'];
    }
  }

  getSelectedPropertyMappingForCurrentConcept() {
    const mapping = this.getMappingForCurrentConcept();
    // I need to find the correct properties associated to active tab
    const activeTab = this.getState().conceptCard.conceptCardState.selectedTab;
    // iterate through the property "properties"
    for (let i=0; i<mapping.length; i++) {
      if (mapping[i].tablink === activeTab) {
        return mapping[i].properties;
      }
    }
  }

  conceptCardPopulatingMessage() {
    const state = this.getState();
    const conceptCard = state.conceptCard;
    const conceptSearch = state.conceptSearch;
    if (conceptCard.cardPopulating) {
      return 'Populating card...please wait.'
    } else if (conceptCard.mvpRemoving) {
      return 'Removing MVP...please wait.'
    } else if (conceptCard.mvpAdding) {
      return 'Adding MVP...please wait.'
    } else if (conceptCard.cardPopulated && !(conceptSearch.selectedConcept.length > 0)) {
      return 'Concept card could not be populated. Please try again.'
    } else {
      return 'Please search and select concept.'
    }
  }
}